caveplot <- function(y1, y2, x1 = 1:length(y1), x2 = x1,
                     delta = diff(range(y1, na.rm = T))/2,
                     coef, xlab, y1lab, y2lab,
                     type = "l", col = 1,
                     fill=T, fill.col=14, fill.dens=-1,
                     keep.mar=F, ...)
{
# R code: Example 1.5
# File: Caveplot.r
#
# AUTHOR    : Henrik Aalborg Nielsen, IMM, DTU (han@imm.dtu.dk (OUTDATED))
#
# y1,x1     : bottom time series (axis on the left), corresponding time points.
# y2,x2     : top time series (axis on the right), corresponding time points.
# delta     : difference between 'curves' in units of y1.
# coef      : the linear transformation by which y2 is transformed
#             missing: coef is obtained from a regression of y1 on y2 
#             (in this case the sampling points must be equal),
#             0: no transformation,
#             numeric of length 2: intercept = coef[1] and slope=coef[2].
# xlab      : label on the x-axis (both the bottom and top one)
# y1lab     : label on left axis (y1)
# y2lab     : label on right axis (y2)
# type      : plot type (may have length 2)
# col       : plot color (may have length 2)
# fill      : for type "l" plots fill above top and below bottom plot?
#             (may have length 2)
# fill.col  : the color used for filling (6 may be more appropriate
#             for grayscale postscript plots)
# fill.dens : density by which to do the filling (may have length 2)
#             -1: solid filling using device-dependent polygon filling
#             algorithm, 0: no filling, pos. integer: density of lines.
# keep.mar  : the function changes the graphics parameter 'mar'.  If this argument
#             is FALSE it will reset it and therefore additional plotting on top
#             of the same plot will be wrong.  Use keep.mar=TRUE in case addtional
#             plotting on top of the cave-plot is needed (this will affect
#             subsequent graphics).
# ...       : parsed to *each* call to lines(), which is used to plot y1 and y2.

  polygon.fill <- function(x, y, ylim, col, density) {
    grp.length <- 1400
    if(length(x) != length(y))
      stop("length of x and y must match")
    if(any(is.na(x)))
      stop("missing values in x don't make sense")
    ## Split in groups of length 'grp.length'
    Ngrps <- ceiling(length(x)/grp.length)
    for(i in 1:Ngrps) {
      idx <- seq(from = grp.length * (i-1),
                 to   = grp.length * i,
                 by   = 1)
      idx <- idx[idx != 0]
      tmp.x <- x[idx]
      tmp.y <- y[idx]
      ## Split at missing values
      tmp.na <- c(ifelse(is.na(tmp.y[1]), 0, -1), diff(is.na(tmp.y)))
      idx.start <- (1:length(tmp.y))[tmp.na == -1]
      tmp.na <- c(tmp.na, ifelse(is.na(tmp.y[length(tmp.y)]), 0, 1))
      idx.end <- (1:(length(tmp.y)+1))[tmp.na == 1] - 1
      if(length(idx.start) != length(idx.end))
        stop("programming error trapped")
      for(i in 1:length(idx.start)) {
        tmp.idx <- idx.start[i]:idx.end[i]
        polygon(x=c(min(tmp.x[tmp.idx]), tmp.x[tmp.idx], max(tmp.x[tmp.idx])),
                y=c(ylim[1], tmp.y[tmp.idx], ylim[1]),
                col=col, density=density, border=F)
      }
    }
  }
  
  if((is.dates(x1) && !is.dates(x2)) || (!is.dates(x1) && is.dates(x2)))
    stop("horizontal axes must be of same type")
  if(any(is.na(x1) | any(is.na(x2))))
    stop("missing values not allowed on horizontal axis")
  if((missing(coef) && length(x1) != length(x2)) ||
     (missing(coef) && any(x1 != x2)))
    stop("argument coef must be supplied when when sampling points are unequal")

  old.mar <- par("mar")
  if(!keep.mar) 
    on.exit(par(mar=old.mar))

  mar <- par("mar")
  mar[4] <- mar[2] # Changed to [2] (from [1]) on Dec. 6, 2006
  mar[3] <- mar[1]
  par(mar = mar)

  if(missing(xlab))
    xlab <- ""
  if(missing(y1lab))
    y1lab <- paste(deparse(substitute(y1)), "(bottom)")
  if(missing(y2lab))
    y2lab <- paste(deparse(substitute(y2)), "(top)")
  if(length(type) == 1) type <- rep(type,2)
  if(length(col) == 1) col <- rep(col,2)
  if(length(fill) == 1) fill <- rep(fill,2)
  if(length(fill.col) == 1) fill.col <- rep(fill.col,2)
  if(length(fill.dens) == 1) fill.dens <- rep(fill.dens,2)
  fill[type != "l"] <- F
  
  py1 <- pretty(y1)
  py2 <- pretty(y2)
  ly1 <- as.character(py1)
  ly2 <- as.character(py2)

  if(missing(coef)) {
    ## Linear cave plot (only when there is some dep. btw. y2(t) and y1(t))
    miss <- (is.na(y1) | is.na(y2))
    coef <- lsfit(x = y2[!miss], y = y1[!miss])$coef
    y2 <- coef[1] + delta + coef[2] * y2
    py2 <- coef[1] + delta + coef[2] * py2
  }
  else if(length(coef) == 1 && coef == 0) {
    ## Standard cave plot (differs slightly from the Lucent version)
    m1 <- mean(y1, na.rm = T)
    m2 <- mean(y2, na.rm = T)
    sd1 <- sqrt(var(y1[!is.na(y1)]))
    sd2 <- sqrt(var(y2[!is.na(y2)]))
    y1 <- (y1 - m1 - delta)/sd1
    py1 <- (py1 - m1 - delta)/sd1
    y2 <- (y2 - m2)/sd2
    py2 <- (py2 - m2)/sd2
  }
  else {
    ## Like linear cave plot, but w/ user supplied linear transformation of y2
    y2 <- coef[1] + delta + coef[2] * y2
    py2 <- coef[1] + delta + coef[2] * py2
  }

  ## More transformations 
  y1min <- min(y1, na.rm = T)
  y1 <- y1 - y1min
  py1 <- py1 - y1min
  y2 <- y2 - y1min
  py2 <- py2 - y1min
  xlim <- range(x1, x2, na.rm = T)
  ylim <- range(y1, y2, na.rm = T)
  y2 <-  - (ylim[2] - y2)
  py2 <-  - (ylim[2] - py2)

  ## Plot #1 (left,bottom)
  plot(x1, y1, type = "n", xlim = xlim, ylim = ylim,
       axes = F, xlab =  xlab, ylab = "")
  if(fill[1])
    polygon.fill(x=x1, y=y1, ylim=ylim, col=fill.col[1], density=fill.dens[1])
  lines(x1, y1, type = type[1], col=col[1], ...)
  box()
  if(is.dates(x1)) {
    xtck.at <- pretty(as.numeric(x1))
    xtck.lab <- format(dates(xtck.at, out = "day mon year"))
    axis(1, at = xtck.at, labels = xtck.lab)
  }
  else axis(1)
  axis(2, at = py1, labels = ly1, srt = 90)
  mtext(y1lab, side = 2, line = 3)

  ## Plot #2 (right, top)
  par(new = T)
  plot(x2, y2, type = "n", xlim = xlim, ylim =  - rev(ylim), 
       axes = F, xlab = "", ylab = "")
  if(fill[2])
    polygon.fill(x=x2, y=y2, ylim=ylim, col=fill.col[2], density=fill.dens[2])
  lines(x2, y2, type = type[2], col=col[2], ...)  
  if(is.dates(x2)) {
    xtck.at <- pretty(as.numeric(x2))
    xtck.lab <- format(dates(xtck.at, out = "day mon year"))
    axis(3, at = xtck.at, labels = xtck.lab)
  }
  else axis(3)
  mtext(xlab, side=3, line=3)
  axis(4, at = py2, labels = ly2, srt = 90)
  mtext(y2lab, side = 4, line = 3)

  cat(paste("Transformation used:", coef[1], "(intercept)", coef[2], "(slope)\n"))
  invisible(return(coef))
}
